/*-----------------------------------------------------------------------------
	[VCE.c]
		Implements the VCE.

	Copyright (C) 2004-2005 Ki

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.
-----------------------------------------------------------------------------*/
#include <stdio.h>
#include <malloc.h>
#include "VCE.h"

#include "ScreenInterface.h"

/*
	[DEV NOTE]

	pbg͓ RGB555 / RGB565 ŕێB
	RGB555 / RGB565 ̂ǂŕێ邩́A
	ʂ̃J[[hČ肷B
*/
static Uint8			_DCC;
static Uint16			_ColorTableAddr;

static Uint16			_Palette333[512];
static Uint16			_Palette[512];
static Uint16			_PaletteBW[512];
static Uint16			_BWPaletteLUT555[512];
static Uint16			_BWPaletteLUT565[512];
static Uint16			_ColorLatch;

static Sint32			_PrevDispW;
static Sint32			_PrevDispH;

static Sint32			_PrevRshift;


static
inline
Uint16
make_RGB555(
	Uint16		color)
{
	return ((color & 0x1c0) << 1) | ((color & 0x038) << 9) | ((color & 0x007) << 2);
}

static
inline
Uint16
make_RGB565(
	Uint16		color)
{
	return ((color & 0x1c0) << 2) | ((color & 0x038) << 10) | ((color & 0x007) << 2);
}



static
inline
void
update_palette()
{
	int		i;
	Uint16	color;
	Sint32	Rshift = SCREEN_GetRshift();

	for (i = 0; i < 512; i++)
	{
		color = _Palette333[i];

		if (i == 0)
		{
			int		j;
			for (j = 0; j < 256; j += 16)
			{
				if (Rshift == 10)		// G = 5-bit ̏ꍇ
				{
					_Palette[j] = make_RGB555(color);
					_PaletteBW[j] = _BWPaletteLUT555[color];
				}
				else					// G = 6-bit ̏ꍇ
				{
					_Palette[j] = make_RGB565(color);
					_PaletteBW[j] = _BWPaletteLUT565[color];
				}
			}
		}
		else if (i == 256)
		{
			int		j;
			for (j = 256; j < 512; j += 16)
			{
				if (Rshift == 10)		// G = 5-bit ̏ꍇ
				{
					_Palette[j] = make_RGB555(color);
					_PaletteBW[j] = _BWPaletteLUT555[color];
				}
				else					// G = 6-bit ̏ꍇ
				{
					_Palette[j] = make_RGB565(color);
					_PaletteBW[j] = _BWPaletteLUT565[color];
				}
			}
		}
		else if ((i & 15) != 0)
		{
			if (Rshift == 10)		// G = 5-bit ̏ꍇ
			{
				_Palette[i] = make_RGB555(color);
				_PaletteBW[i] = _BWPaletteLUT555[color];
			}
			else					// G = 6-bit ̏ꍇ
			{
				_Palette[i] = make_RGB565(color);
				_PaletteBW[i] = _BWPaletteLUT565[color];
			}
		}
	}
}

/*
	[write_palette]

	convert GRB333 --> RGB555 / RGB565
	and write to the palette.

	basic logic:

	Uint8 G = (color & 0x1c0) >> 6;
	Uint8 R = (color & 0x038) >> 3;
	Uint8 B = (color & 0x007) >> 0;

	Rshift = 10;
	Gshift = 5;
	Bshift = 0;

	_Palette[addr]	= R << (Rshift+2);
	_Palette[addr] |= G << (Gshift+2);
	_Palette[addr] |= B << (Bshift+2);
*/
static
inline
void
write_palette(
	Uint16		addr,
	Uint16		color)
{
	Uint32 Rshift = SCREEN_GetRshift();

	_Palette333[addr] = color;

	// ʂ̂aooύXĂpbgXV 
	if (Rshift != _PrevRshift)
	{
		_PrevRshift = Rshift;
		update_palette();
	}
	else
	{
		if (addr == 0)
		{
			int		i;
			for (i = 0; i < 256; i += 16)
			{
				if (Rshift == 10)		// G = 5-bit ̏ꍇ
				{
					_Palette[i] = make_RGB555(color);
					_PaletteBW[i] = _BWPaletteLUT555[color];
				}
				else					// G = 6-bit ̏ꍇ
				{
					_Palette[i] = make_RGB565(color);
					_PaletteBW[i] = _BWPaletteLUT565[color];
				}
			}
		}
		else if (addr == 256)
		{
			int		i;
			for (i = 256; i < 512; i += 16)
			{
				if (Rshift == 10)		// G = 5-bit ̏ꍇ
				{
					_Palette[i] = make_RGB555(color);
					_PaletteBW[i] = _BWPaletteLUT555[color];
				}
				else					// G = 6-bit ̏ꍇ
				{
					_Palette[i] = make_RGB565(color);
					_PaletteBW[i] = _BWPaletteLUT565[color];
				}
			}
		}
		else if ((addr & 15) != 0)
		{
			if (Rshift == 10)		// G = 5-bit ̏ꍇ
			{
				_Palette[addr] = make_RGB555(color);
				_PaletteBW[addr] = _BWPaletteLUT555[color];
			}
			else					// G = 6-bit ̏ꍇ
			{
				_Palette[addr] = make_RGB565(color);
				_PaletteBW[addr] = _BWPaletteLUT565[color];
			}
		}
	}
}


/*
	[read_palette]

	convert RGB555 / RGB565 --> GRB333

	basic logic:

	Uint8 R = (_Palette[addr] & 0x7c00) >> 10;
	Uint8 G = (_Palette[addr] & 0x03e0) >> 5;
	Uint8 B = (_Palette[addr] & 0x001f) >> 0;

	G >>= 2; R >>=	2; B >>= 2;

	return (G << 6) | (R << 3) | B;

	 0RRR RRGG GGGB BBBB
   & 0111 0011 1001 1100  (0x739c)
   = 0RRR 00GG G00B BB00

	 0000 000G GGRR RBBB
*/
static
inline
Uint16
read_palette(
	Uint16		addr)
{
	return _Palette333[addr];
}


/*-----------------------------------------------------------------------------
	[VCE_Write]
		ubdւ̏ݓLq܂B
-----------------------------------------------------------------------------*/
void
VCE_Write(
	Uint32		regNum,
	Uint8		data)
{
	switch (regNum & 7)
	{
		case 0: /* Rg[WX^ k */
			// (hbgNbNw肷) 
			_DCC = data & 0x87;
			return;

		case 1: /* Rg[WX^ g */
			return;

		case 2:	/* J[e[uAhX k */
			_ColorTableAddr &= 0x100;
			_ColorTableAddr |= data;
			return;

		case 3:	/* J[e[uAhX g */
			_ColorTableAddr &= 0xff;
			_ColorTableAddr |= (data & 1) << 8;
			return;

		case 4:	/* J[e[uf[^Cg k */
			_ColorLatch &= 0x100;
			_ColorLatch |= data;
			return;

		case 5:	/* J[e[uf[^Cg g */
			_ColorLatch &= 0xff;
			_ColorLatch |= (data & 1) << 8;
			write_palette(_ColorTableAddr, _ColorLatch);
			_ColorTableAddr = (_ColorTableAddr + 1) & 0x1ff;
			return;
	}
}


/*-----------------------------------------------------------------------------
	[VCE_Read]
		ubd̓ǂݏoLq܂D
-----------------------------------------------------------------------------*/
Uint8
VCE_Read(
	Uint32		regNum)
{
	Uint8		ret;

	switch (regNum)
	{
		case 4:
			return (Uint8)read_palette(_ColorTableAddr);
		case 5:
			ret = (Uint8)(read_palette(_ColorTableAddr) >> 8) | 0xfe;
			_ColorTableAddr = (_ColorTableAddr + 1) & 0x1ff;
			return ret;
	}

	/* don't return 0xff */
	return 0;
//	return 0xff;
}

void
VCE_EncodeLine(
	Uint16*			pDst,		// out: encoded pixels (RGB555/565)
	const Uint8*	pSrc,		// in:	palette index (8-bit)
	const Uint8*	pSpBg,		// in:	SP / BG bit array
	Sint32			width)		// in:	scanline width (W̔{Ŏw肷) 
{
	Uint32*		pdwSpBg = (Uint32*)pSpBg;
	Uint32 Rshift = SCREEN_GetRshift();

	// ʂ̂aooύXĂpbgXV 
	if (Rshift != _PrevRshift)
	{
		update_palette();
		_PrevRshift = Rshift;
	}

	width /= 8;
	if (_DCC & 0x80)
	{
		while (width-- > 0)
		{
			*pdwSpBg++ &= 0x01010101;
			*pdwSpBg++ &= 0x01010101;

			pDst[0] = _PaletteBW[pSrc[0]+(pSpBg[0]<<8)];
			pDst[1] = _PaletteBW[pSrc[1]+(pSpBg[1]<<8)];
			pDst[2] = _PaletteBW[pSrc[2]+(pSpBg[2]<<8)];
			pDst[3] = _PaletteBW[pSrc[3]+(pSpBg[3]<<8)];
			pDst[4] = _PaletteBW[pSrc[4]+(pSpBg[4]<<8)];
			pDst[5] = _PaletteBW[pSrc[5]+(pSpBg[5]<<8)];
			pDst[6] = _PaletteBW[pSrc[6]+(pSpBg[6]<<8)];
			pDst[7] = _PaletteBW[pSrc[7]+(pSpBg[7]<<8)];

			pDst  += 8;
			pSrc  += 8;
			pSpBg += 8;
		}
	}
	else
	{
#if defined(__GNUC__) && defined(USE_INLINE_ASM)
		width /= 2;
		while (width-- > 0)
		{
			__asm__ (
				"andl		$0x01010101,  (%2)				\n\t"
				"andl		$0x01010101, 4(%2)				\n\t"
				"andl		$0x01010101, 8(%2)				\n\t"
				"andl		$0x01010101,12(%2)				\n\t"
				"movd		(%1),  %%mm0					\n\t"
				"movd		(%2),  %%mm1					\n\t"
				"punpcklbw	%%mm1, %%mm0					\n\t"

				// [0], [1]
				"movd		%%mm0, %%eax					\n\t"

				"movzwl		%%ax,  %%ebx					\n\t"
				"movw		__Palette(%%ebx,%%ebx), %%bx	\n\t"
				"movw		%%bx,  (%0)						\n\t"

				"shrl		$16,   %%eax					\n\t"
				"movw		__Palette(%%eax,%%eax), %%ax	\n\t"
				"movw		%%ax,  2(%0)					\n\t"

				"psrlq		$32,   %%mm0					\n\t"

				// [2], [3]
				"movd		%%mm0, %%eax					\n\t"

				"movzwl		%%ax,  %%ebx					\n\t"
				"movw		__Palette(%%ebx,%%ebx), %%bx	\n\t"
				"movw		%%bx,  4(%0)					\n\t"

				"shrl		$16,   %%eax					\n\t"
				"movw		__Palette(%%eax,%%eax), %%ax	\n\t"
				"movw		%%ax,  6(%0)					\n\t"

				"movd		4(%1), %%mm0					\n\t"
				"movd		4(%2), %%mm1					\n\t"
				"punpcklbw	%%mm1, %%mm0					\n\t"

				// [4], [5]
				"movd		%%mm0, %%eax					\n\t"

				"movzwl		%%ax,  %%ebx					\n\t"
				"movw		__Palette(%%ebx,%%ebx), %%bx	\n\t"
				"movw		%%bx,  8(%0)					\n\t"

				"shrl		$16,   %%eax					\n\t"
				"movw		__Palette(%%eax,%%eax), %%ax	\n\t"
				"movw		%%ax,  10(%0)					\n\t"

				"psrlq		$32,   %%mm0					\n\t"

				// [6], [7]
				"movd		%%mm0, %%eax					\n\t"

				"movzwl		%%ax,  %%ebx					\n\t"
				"movw		__Palette(%%ebx,%%ebx), %%bx	\n\t"
				"movw		%%bx,  12(%0)					\n\t"

				"shrl		$16,   %%eax					\n\t"
				"movw		__Palette(%%eax,%%eax), %%ax	\n\t"
				"movw		%%ax,  14(%0)					\n\t"

				"movd		8(%1),  %%mm0					\n\t"
				"movd		8(%2),  %%mm1					\n\t"
				"punpcklbw	%%mm1, %%mm0					\n\t"

				// [0], [1]
				"movd		%%mm0, %%eax					\n\t"

				"movzwl		%%ax,  %%ebx					\n\t"
				"movw		__Palette(%%ebx,%%ebx), %%bx	\n\t"
				"movw		%%bx,  16(%0)					\n\t"

				"shrl		$16,   %%eax					\n\t"
				"movw		__Palette(%%eax,%%eax), %%ax	\n\t"
				"movw		%%ax,  18(%0)					\n\t"

				"psrlq		$32,   %%mm0					\n\t"

				// [2], [3]
				"movd		%%mm0, %%eax					\n\t"

				"movzwl		%%ax,  %%ebx					\n\t"
				"movw		__Palette(%%ebx,%%ebx), %%bx	\n\t"
				"movw		%%bx,  20(%0)					\n\t"

				"shrl		$16,   %%eax					\n\t"
				"movw		__Palette(%%eax,%%eax), %%ax	\n\t"
				"movw		%%ax,  22(%0)					\n\t"

				"movd		12(%1), %%mm0					\n\t"
				"movd		12(%2), %%mm1					\n\t"
				"punpcklbw	%%mm1, %%mm0					\n\t"

				// [4], [5]
				"movd		%%mm0, %%eax					\n\t"

				"movzwl		%%ax,  %%ebx					\n\t"
				"movw		__Palette(%%ebx,%%ebx), %%bx	\n\t"
				"movw		%%bx,  24(%0)					\n\t"

				"shrl		$16,   %%eax					\n\t"
				"movw		__Palette(%%eax,%%eax), %%ax	\n\t"
				"movw		%%ax,  26(%0)					\n\t"

				"psrlq		$32,   %%mm0					\n\t"

				// [6], [7]
				"movd		%%mm0, %%eax					\n\t"

				"movzwl		%%ax,  %%ebx					\n\t"
				"movw		__Palette(%%ebx,%%ebx), %%bx	\n\t"
				"movw		%%bx,  28(%0)					\n\t"

				"shrl		$16,   %%eax					\n\t"
				"movw		__Palette(%%eax,%%eax), %%ax	\n\t"
				"movw		%%ax,  30(%0)					\n\t"
			:
			: "r" (pDst), "r" (pSrc), "r" (pSpBg)
			: "%eax", "%ebx");

			pDst  += 16;
			pSrc  += 16;
			pSpBg += 16;
		}
		__asm__ ("emms");
#else
		while (width-- > 0)
		{
			*pdwSpBg++ &= 0x01010101;
			*pdwSpBg++ &= 0x01010101;

			pDst[0] = _PaletteBW[pSrc[0]+(pSpBg[0]<<8)];
			pDst[1] = _PaletteBW[pSrc[1]+(pSpBg[1]<<8)];
			pDst[2] = _PaletteBW[pSrc[2]+(pSpBg[2]<<8)];
			pDst[3] = _PaletteBW[pSrc[3]+(pSpBg[3]<<8)];
			pDst[4] = _PaletteBW[pSrc[4]+(pSpBg[4]<<8)];
			pDst[5] = _PaletteBW[pSrc[5]+(pSpBg[5]<<8)];
			pDst[6] = _PaletteBW[pSrc[6]+(pSpBg[6]<<8)];
			pDst[7] = _PaletteBW[pSrc[7]+(pSpBg[7]<<8)];

			pDst  += 8;
			pSrc  += 8;
			pSpBg += 8;
		}
#endif
	}
}


BOOL
VCE_IsHiresolutionMode()
{
	return ((_DCC & 6) == 6);
}


/*-----------------------------------------------------------------------------
	[VCE_Init]
		ubd܂B
-----------------------------------------------------------------------------*/
Sint32
VCE_Init()
{
	int			i;
	float		b;
	float		g;
	float		r;
	float		bw;
	int			ibw;

	for (i = 0; i < 512; ++i)
	{
		_Palette333[i] = 0x1ff;
		_Palette[i] = 0xffff;
		_PaletteBW[i] = 0xffff;

		b = (float)(i & 7);
		r = (float)((i >> 3) & 7);
		g = (float)((i >> 6) & 7);

		bw = (r * 0.299 + g * 0.587 + b * 0.114) * 31.0 / 7.0;

		ibw = (int)bw;
		_BWPaletteLUT555[i] = ibw | (ibw << 5) | (ibw << 10);
		_BWPaletteLUT565[i] = ibw | (ibw << 6) | (ibw << 11);
	}

	_DCC = 0;
	_ColorTableAddr = 0;
	_ColorLatch = 0;
	_PrevDispW = 0;
	_PrevDispH = 0;
	_PrevRshift = 0;

	return 0;
}


/*-----------------------------------------------------------------------------
	[VCE_Deinit]
		ubdj܂D
-----------------------------------------------------------------------------*/
void
VCE_Deinit()
{
	memset(_Palette333, 0, sizeof(_Palette333));
	memset(_Palette, 0, sizeof(_Palette));
	_DCC = 0;
	_ColorTableAddr = 0;
	_ColorLatch = 0;
	_PrevDispW = 0;
	_PrevDispH = 0;
	_PrevRshift = 0;
}


// save variable
#define SAVE_V(V)	if (fwrite(&V, sizeof(V), 1, p) != 1)	return FALSE
#define LOAD_V(V)	if (fread(&V, sizeof(V), 1, p) != 1)	return FALSE
// save array
#define SAVE_A(A)	if (fwrite(A, sizeof(A), 1, p) != 1)	return FALSE
#define LOAD_A(A)	if (fread(A, sizeof(A), 1, p) != 1)		return FALSE
/*-----------------------------------------------------------------------------
	[SaveState]
		Ԃt@Cɕۑ܂B 
-----------------------------------------------------------------------------*/
BOOL
VCE_SaveState(
	FILE*		p)
{
	if (p == NULL)
		return FALSE;

	SAVE_V(_DCC);
	SAVE_V(_ColorTableAddr);

	SAVE_A(_Palette333);
	SAVE_A(_Palette);

	SAVE_V(_ColorLatch);

	SAVE_V(_PrevDispW);
	SAVE_V(_PrevDispH);

	SAVE_V(_PrevRshift);

	return TRUE;
}


/*-----------------------------------------------------------------------------
	[LoadState]
		Ԃt@Cǂݍ݂܂B 
-----------------------------------------------------------------------------*/
BOOL
VCE_LoadState(
	FILE*		p)
{
	if (p == NULL)
		return FALSE;

	LOAD_V(_DCC);
	LOAD_V(_ColorTableAddr);

	LOAD_A(_Palette333);
	LOAD_A(_Palette);

	LOAD_V(_ColorLatch);

	LOAD_V(_PrevDispW);
	LOAD_V(_PrevDispH);

	LOAD_V(_PrevRshift);

	return TRUE;
}

#undef SAVE_V
#undef SAVE_A
#undef LOAD_V
#undef LOAD_A
